(*
 * LANGUAGE    : ANS Forth
 * PROJECT     : Forth Environments
 * DESCRIPTION : A simple Telnet client
 * CATEGORY    : Example 
 * AUTHOR      : Jeffrey R. Fox
 * LAST CHANGE : December 15, 2002, Marcel Hendrix 
 *)


	NEEDS -miscutil
	NEEDS -sockets

	REVISION -telnet " Telnet client       Version 1.01 "

	PRIVATES

DOC
(*
	Author: Jeffrey R. Fox Jeffrey.Fox@wcom.com
		Site Controller/Access Server Development Team
		Worldcom, Colorodo Springs, CO

	A demonstration of a polled socket interface
	============================================

	Demonstrates using socket API functions to implement a simple
	line-oriented Telnet client. Takes one|two parameters, as
	follows:

	      <port#> telnet host_addr|host_name

	Connects to the specified server, allowing two-way communication and
	providing minimalistic telnet negotiation.

	NOTES:

	   This program supports both Windows and Linux sockets and requires a
	   TCP/IP networking (or another TCP/IP stack) to be installed and 
	   configured prior to use. You will also need to have a TCP/IP 
	   connection established or else have auto-dialing enabled.

	   Inspired in part by a sendmail program written by Jim Blaney, 1996 
	   in "Mastering Windows NT Programming."

	REFERENCES:

	Read the RFC docs.
	If you are going to read (parts of) a book, make it "TCP/IP Illustrated, 
	Volume 1 - The Protocols" by W. Richard Stevens.
*)
ENDDOC

#23 =: IPPORT_TELNET

VARIABLE tn-debug? 	tn-debug? ON			-- turn this on for some debugging info

CREATE tn-outbuf 	PRIVATE #256 CHARS ALLOT	-- I/O buffers for a line oriented interface
CREATE tn-inbuf  	PRIVATE #256 CHARS ALLOT 
CREATE server		PRIVATE #256 CHARS ALLOT	-- {name | address in dotted notation} of the 
							-- server or host system we need to connect to
0 VALUE port#		PRIVATE				-- port#, normally IPPORT_TELNET (23) 
0 VALUE lsocket 	PRIVATE				-- socket to communicate through	
CREATE  charbuff 0 ,	PRIVATE				-- count + 3 characters

: buffer-flush?	( -- )		 tn-outbuf C@ #254 = IF  tn-outbuf COUNT lsocket WRITE-SOCKET  tn-outbuf C0!  ENDIF ;P
: char-append 	( c 'buffer -- ) SWAP charbuff C!  charbuff 1 ROT PLACE+  buffer-flush? ;P

: tn-putchar ( char -- )
	DUP ^M = IF  DROP tn-outbuf COUNT +CR lsocket WRITE-SOCKET  tn-outbuf C0!
	       ELSE  tn-outbuf char-append  
	      ENDIF ;P

DOC
(*
   The following are the defined TELNET commands.  Note that these codes
   and code sequences have the indicated meaning only when immediately
   preceded by an IAC byte (#iac = 255).

      NAME               CODE              MEANING

      EOF                 236    end-of-file
      SUSP                237    suspend current process (job control)
      ABORT               238    abort process
      EOR                 239    end of record
      SE                  240    End of subnegotiation parameters.
      NOP                 241    No operation.
      Data Mark           242    The data stream portion of a Synch.
                                 This should always be accompanied
                                 by a TCP Urgent notification.
      Break               243    NVT character BRK.
      Interrupt Process   244    The function IP.
      Abort output        245    The function AO.
      Are You There       246    The function AYT.
      Erase character     247    The function EC.
      Erase Line          248    The function EL.
      Go ahead            249    The GA signal.
      SB                  250    Indicates that what follows is
                                 subnegotiation of the indicated
                                 option.
      WILL (option code)  251    Indicates the desire to begin
                                 performing, or confirmation that
                                 you are now performing, the
                                 indicated option.
      WON'T (option code) 252    Indicates the refusal to perform,
                                 or continue performing, the
                                 indicated option.
      DO (option code)    253    Indicates the request that the
                                 other party perform, or
                                 confirmation that you are expecting
                                 the other party to perform, the
                                 indicated option.
      DON'T (option code) 254    Indicates the demand that the
                                 other party stop performing,
                                 or confirmation that you are no
                                 longer expecting the other party
                                 to perform, the indicated option.
      IAC                 255    Data Byte 255. "Interpret As Command"
*)
ENDDOC

-- Telnet command codes
-- note: the #iac means the next byte and perhaps more are telnet commands
-- and should be considered separate from the rest of the (default 7-bit) ASCII
-- stream.

#255 =: #iac		PRIVATE	-- Interpret As Command
#247 =: #echar  	PRIVATE	-- Erase Character
#251 =: #will   	PRIVATE	-- option negotiation: sender wants to enable option
#252 =: #won't  	PRIVATE	-- option negotiation: sender wants to disable option
#253 =: #do     	PRIVATE	-- option negotiation: sender wants receiver to enable option
#254 =: #don't  	PRIVATE	-- option negotiation: sender wants receiver to disable option

-- options:

   1 =: #echo		PRIVATE	
   3 =: #sga		PRIVATE
   5 =: #status		PRIVATE	
 #34 =: #linemode	PRIVATE	
 #24 =: #terminaltype	PRIVATE	

-- The telnet negotiation is implemented as a state machine to filter out the Telnet commands 
-- from the incoming stream.  The operation of this state machine is the only thing that truely 
-- characterizes Telnet from a raw data stream.

-- STATES:
--    TN-NORMAL TN-DO-COMMAND TN-NEGOTIATE

DEFER :filter	 	PRIVATE	-- the present state stored as an execution token
DEFER :normal ( -- )	PRIVATE	-- go to normal state

-- Send out 3 byte telnet negotiation code
: tn-send-neg ( topic opinion -- )
	     #iac  charbuff     C! 
	( opinion) charbuff 1+  C! 
	(   topic) charbuff 2+  C!
	charbuff 3 lsocket WRITE-SOCKET ;P

0 VALUE tn-command	PRIVATE

: .debug ( c-addr u -- ) 
	DUP tn-debug? @ AND IF  CR >BOLD< ." HOST [" .TIME ." ] " TYPE >BOLD< SPACE EXIT  ENDIF
	2DROP ;P

-- We always want to do local echoing to avoid network waste on chatter
: negotiate-echo  ( -- )
	CASE tn-command
	  #will  OF  #echo #don't tn-send-neg  S" will echo?  DON'T" .debug  ENDOF
	  #do    OF  #echo #won't tn-send-neg  S" do echo?    WON'T" .debug  ENDOF
	  #won't OF  #echo #don't tn-send-neg  S" won't echo? DON'T" .debug  ENDOF
	  #don't OF  #echo #won't tn-send-neg  S" don't echo? WON'T" .debug  ENDOF
   	ENDCASE ;P

-- We don't want to send go ahead characters -- that's history.
: negotiate-ga  ( -- )
	CASE tn-command 
          #will  OF  #sga  #do    tn-send-neg  S" will suppress-go-ahead?  DO"    .debug  ENDOF
          #do    OF  #sga  #will  tn-send-neg  S" do suppress-go-ahead?    WILL"  .debug  ENDOF
          #won't OF  #sga  #don't tn-send-neg  S" won't suppress-go-ahead? DON'T" .debug  ENDOF
          #don't OF  #sga  #won't tn-send-neg  S" don't suppress-go-ahead? WON'T" .debug  ENDOF
	ENDCASE ;P

-- We don't do unsupported options, obviously -- in this case that is about everything
: negotiate-unsupported-option ( option -- )
	CASE tn-command 
          #will  OF        #don't tn-send-neg  S" will ??  DON'T" .debug  ENDOF
          #do    OF        #won't tn-send-neg  S" do ??    WON'T" .debug  ENDOF
          #won't OF        #don't tn-send-neg  S" won't ?? DON'T" .debug  ENDOF
          #don't OF        #won't tn-send-neg  S" will ??  DON'T" .debug  ENDOF
	ENDCASE ;P

-- Process a WILL/WON'T/DO/DON'T topic
-- TRUE means treat as normal char
: tn-negotiate ( char -- 0 0 )
	CASE
          #echo OF  negotiate-echo  ENDOF
          #sga  OF  negotiate-ga    ENDOF
          ( default) negotiate-unsupported-option  0
	ENDCASE 
	0 0 :normal ;P

: >negotiate ( -- ) ['] tn-negotiate IS :filter ;P

-- Process a telnet command
-- TRUE if char is to be treated normally
: tn-do-command ( char -- char f ) 	
	CASE DUP TO tn-command 
	  #won't OF  >negotiate 0 0    ENDOF
     	  #don't OF  >negotiate 0 0    ENDOF
     	  #will  OF  >negotiate 0 0    ENDOF
     	  #do    OF  >negotiate 0 0    ENDOF
     	  #iac   OF  #iac TRUE :normal ENDOF
          ( default)    0    0 :normal	( unsupported command eaten )
   	ENDCASE ;P

-- Process a normal character, look for IAC
-- TRUE if normal char
: tn-normal ( char -- char f )		
	DUP #iac <> DUP 0= IF  ['] tn-do-command IS :filter ENDIF ;P

: (normal)  ['] tn-normal IS :filter ;P  ' (normal) IS :normal

-- filter Telnet commands out of stream; reply to all negotiation requests
: tn-filter ( c-addr u -- )
	BOUNDS ?DO  
		    I C@ :filter
		       IF  DUP ^M = IF  DROP CR
			          ELSE  DUP  ^J = IF  DROP
				  		ELSE  EMIT
				     	       ENDIF
			         ENDIF
		     ELSE  DROP
		    ENDIF
	      LOOP ;P


-- Poll for incoming data -- filter and print to console
: tn-poll ( -- ) lsocket tn-inbuf #255 READ-SOCKET tn-filter ;P

TRUE VALUE echo? PRIVATE -- standard, telnet host starts in noecho, line mode

: tn-loop ( -- )
	BEGIN	
	  tn-poll
	  EKEY? IF  EKEY DUP ESC = IF  DROP EXIT  ENDIF
		    echo? IF  DUP ^M = IF  CR  ELSE  DUP EMIT  ENDIF  ENDIF
		    tn-putchar
	     ENDIF
	AGAIN ;P

: tn-init ( port# "server" -- )
	TO port#  /PARSE server PACK DROP 
	server COUNT port# OPEN-SERVICE TO lsocket 
	lsocket FALSE BLOCKING-MODE 	( no blocking for telnet)
	0 SET-SOCKET-TIMEOUT		( interactive mode )
	tn-outbuf C0!			( null output buffer to start )
	:normal ;P			( reset Telnet filter )

: tn-exit ( -- )
	CR ." Closing telnet on user request" 
	lsocket CLOSE-SOCKET ;P

: TELNET ( port# "server" -- )
	tn-init  tn-loop  tn-exit ;

:ABOUT  CR ." Usage: port# TELNET host_name|host_IP_address" 
	CR
   	CR ." Example: #23 TELNET iae.nl" ;

                .ABOUT -telnet CR
		DEPRIVE
	                              (* End of Source *)