(*
 * LANGUAGE    : ANS Forth 
 * PROJECT     : Forth Environments
 * DESCRIPTION : Input and output sockets accessible with GETCH and PUTCH
 * CATEGORY    : Utility
 * AUTHOR      : Marcel Hendrix 
 * LAST CHANGE : Sunday, December 22, 2002 7:55 PM, cloned from streams.frt
 *)



	NEEDS -miscutil
	NEEDS -buffers
	NEEDS -sockets

	REVISION -sstream " Socket streams      Version 1.01 "

	PRIVATES


-- Predefine socketbuflen if you want larger (or smaller!) default buffer size


[UNDEFINED] socketbuflen [IF] #4096 VALUE socketbuflen [THEN]


DOC Streams 
(* 
 Socket streams have a buffered input and output channel. Buffers are automatically
 allocated when OPENing, deallocated when CLOSEing. A stream uses the TO-concept.
 
 The following ( TO ) operations are possible: ( stream is a socket stream identifier)

  $OPEN-I/O	stream 			\ ( c-addr u port# -- )  aborts if error
  SCLOSE-I/O    stream 			\ aborts if error

  SGETCH	stream 			\ ( -- byte | eof )
  SPUTCH	stream 			\ ( byte -- )
*)
ENDDOC


-- Access the fields in the pfa of a sstream

:INLINE 'socket	    0 CELL[] ;	PRIVATE	-- socket descriptor

:INLINE inbuf	    1 CELL[] ;	PRIVATE -- buffer address
:INLINE inbuflen    2 CELL[] ;	PRIVATE -- allocated size
:INLINE insize	    3 CELL[] ;	PRIVATE -- real size
:INLINE inptr	    4 CELL[] ;	PRIVATE -- buffer pointer

:INLINE outbuf	    5 CELL[] ;	PRIVATE -- buffer address
:INLINE outbuflen   6 CELL[] ;	PRIVATE -- allocated size
:INLINE outsize	    7 CELL[] ;	PRIVATE -- real size
:INLINE outptr	    8 CELL[] ;	PRIVATE -- buffer pointer


: !OPEN-I/O ( c-addr u port#  addr  -- )
		>S
		OPEN-SERVICE  S 'socket ! 
		S  outptr  OFF  
		S  outsize OFF  
		S  outbuf  GET-BUF
		S  inptr   OFF   
		S  insize  OFF  
		S  inbuf   GET-BUF 
		S> 'socket @ FALSE BLOCKING-MODE ; PRIVATE

-- Write the remaining data in the output buffer and close files & buffers.

: !CLOSE-I/O	>S 				\ ( addr -- )
		 S 'socket @ 0= IF  -S EXIT  ENDIF
		 S outbuf @  S outptr @  S 'socket @  WRITE-SOCKET
		 S inbuf  FREE-BUF  
		 S outbuf FREE-BUF
		 S 'socket @ CLOSE-SOCKET  
		S> 'socket OFF ; PRIVATE

-1 =: EOF


-- Get next byte from input socket 

: !GETCH 	DUP >S				\ ( addr -- c | EOF )
		DUP inptr @  SWAP insize @ =	\ No more bytes left in buffer?
		     IF
			S 'socket  @  
		        S inbuf    @  
			S inbuflen @  
			READ-SOCKET
	        	DUP 0= IF 2DROP -S 
				  EOF EXIT  	\ Exit with EOF if empty.
		            ENDIF
			( -- c-addr u ) 
			S insize !  DROP
	        	S inptr OFF
		  ENDIF
		S>
		DUP inptr @  OVER inbuf @ + C@	\ Fetch byte from buffer
		SWAP 1 SWAP inptr +! ; PRIVATE	\ increment index


-- put byte in output socket.

: !PUTCH 	DUP >S				\ ( c addr -- )	
		DUP outptr @ SWAP outbuflen @ =	\ buffer full?
		     IF
			S outbuf    @  
			S outbuflen @  
			S 'socket   @  
			WRITE-SOCKET 
			S outptr  OFF
			S outsize OFF
		  ENDIF
		S DUP 
		outptr @  SWAP outbuf @ +  C!	\ Store byte into buffer
		1 S  outptr  +! 
		1 S> outsize +! ; PRIVATE	\ increment index and size

: !FLUSH 	>S				\ ( addr -- )	
		 S outbuf  @  
		 S outsize @  
		 S 'socket @  WRITE-SOCKET 
		 S inptr   OFF 
		 S insize  OFF
		 S outptr  OFF 
		S> outsize OFF ; PRIVATE	

USER %socket	PRIVATE	  -1 %socket !

: $OPEN-I/O	0 %socket ! ; IMMEDIATE
: SCLOSE-I/O	1 %socket ! ; IMMEDIATE
: SGETCH	2 %socket ! ; IMMEDIATE
: SPUTCH	3 %socket ! ; IMMEDIATE
: SFLUSH	4 %socket ! ; IMMEDIATE
: @SOCKET	5 %socket ! ; IMMEDIATE

: SSTREAM	CREATE	IMMEDIATE		\ STREAM "name"  
			0 ,			\ socket descriptor
			  0 ,			\   input buffer address
			  socketbuflen ,  	\   keep these two together
			0 ,			\ real input size
			0 ,			\ input buffer pointer
			  0 ,			\   output buffer address
			  socketbuflen ,	\   keep these two together
			0 ,			\ real output size
			0 ,			\ output buffer pointer
		FORGET>	!CLOSE-I/O
		DOES>	ALITERAL
			%socket @  -1 %socket !
			CASE
			   0 OF ['] !OPEN-I/O      NOW? ENDOF
			   1 OF ['] !CLOSE-I/O     NOW? ENDOF
 			   2 OF ['] !GETCH         NOW? ENDOF
			   3 OF ['] !PUTCH         NOW? ENDOF
			   4 OF ['] !FLUSH	   NOW? ENDOF
			   5 OF ['] @		   NOW? ENDOF
			   DUP ABORT" SSTREAM : illegal action"
			ENDCASE ;

:ABOUT	CR ." -- Buffered socket I/O" 
	CR
	CR ." xxx TO socketbuflen                  \ set size of SSTREAMs defined next"
	CR ."                                      \ (default uses " socketbuflen DEC. ." bytes)"
	CR ." SSTREAM str                          \ create a new buffered socket I/O stream str"
	CR .~ ( c-addr u port# -- ) $OPEN-I/O str  \ attach named service via port to stream `str'~
	CR ." SCLOSE-I/O str                       \ close `str'" 
	CR ." GETCH str ( -- c|eof )               \ return next byte from `str', EOF = -1"
	CR ." ( c -- ) PUTCH str                   \ write byte to `str'"
	CR ." SFLUSH str ( -- )                    \ flush stream socket - write out and reset in"
	CR ." @SOCKET str ( -- socket )            \ get stream socket (for IOCTL)"
	CR ." EOF ( -- -1 )                        \ returned by GETCH if end of file" ;

	DEPRIVE
	nesting @ 1 = [IF] .ABOUT -sstream CR [THEN]

				(* End of SSTREAMS *)