(*
 * LANGUAGE    : ANS Forth
 * PROJECT     : Forth Environments
 * DESCRIPTION : Send e-mail and post to usenet groups
 * CATEGORY    : Utility 
 * AUTHOR      : Marcel Hendrix 
 * LAST CHANGE : December 15, 2002, Marcel Hendrix 
 * LAST CHANGE : Sunday, December 22, 2002 5:49 PM, Marcel Hendrix; added POST
 * LAST CHANGE : Tuesday, December 24, 2002 11:25 AM, Marcel Hendrix; multiple recipients
 *)

	NEEDS -miscutil
	NEEDS -sstream

	REVISION -pmail " Send mail & news    Version 1.05 "

	PRIVATES

DOC
(*
   Socket API functions to perform email transmission using the SMTP protocol. 
   Takes three parameters, as follows:

            sendmail  to_addr "subject" "messagefile.txt"

   Connect to the (internally specified) mail server, use to_addr (and an internally 
   preset from_addr) in the transmitted message header, then send real name (internal), 
   subject, repeated to-field and each line of the messagefile (text file) as the 
   email message body. As a convenience, the program sends a copy of all outgoing mail 
   to the sender.

   Message data is prepended with extra info to put anti-spam software at ease.

   Posting news articles is very similar to sending mail -- another port# and some
   required headings inside the data part. Therefore I've added code here instead 
   of putting this functionality in a separate file.
*)
ENDDOC

-- standard tools --------------------------------------------------------------------------------------------------------

SSTREAM [sock]		PRIVATE
CREATE  perr 		PRIVATE	3 CHARS ALLOT 		-- buffers response codes from server
CREATE  response$	PRIVATE	#256 CHARS ALLOT	-- buffers response from server
0 VALUE #resp		PRIVATE				-- response size
0 VALUE file		PRIVATE

VARIABLE debug? debug? OFF

: <$+>    ( c-addr1 u1 c-addr2 u2 -- c-addr3 u3 ) S" <" 2SWAP $+ S" >" $+ $+ ;P
: .debug  ( c-addr u -- ) DUP debug? @ AND IF  ^M EMIT '[' EMIT .TIME ." ] " TYPE EXIT  ENDIF 2DROP ;P
: <semit> ( char -- )     DUP EOF = IF  DROP ." <EOF>"  ELSE  EMIT  ENDIF ;P
: ?emit   ( char -- )     debug? @ IF  <semit> EXIT  ENDIF DROP ;P
: FSEND$  ( c-addr u -- ) 0 ?DO  C@+ SPUTCH [sock]  LOOP DROP ;P ( do a timely flush! )
: SEND$   ( c-addr u -- ) FSEND$  SFLUSH [sock] ;P
: CHAR1   ( -- c ) 	  BEGIN  SGETCH [sock] DUP EOF =  WHILE  DROP  REPEAT ;P
: !char	  ( c -- )	  response$ #resp +  C!  #resp 1+ #255 AND TO #resp ;P

: EAT-ALL ( -- )
	CLEAR #resp
	CHAR1 DUP ?emit !char 
	BEGIN  SGETCH [sock] DUP ?emit DUP !char EOF =  UNTIL ;P

-- Errors are in perr. It is not necessary to translate them.
: .REPORT-ERRORS ( bool -- ) 
	0= ?EXIT
	CR ." Error " perr 3 TYPE SPACE response$ #resp TYPE
	TRUE ABORT" protocol error" ;P

: cmd ( c-addr u -- )
	+CR 2DUP .debug SEND$
	CHAR1         DUP ?emit  perr           C!
	SGETCH [sock] DUP ?emit  perr   CHAR+   C!
	SGETCH [sock] DUP ?emit  perr 2 CHARS + C!
	perr C@ '3' > >S ( > 4xx is protocol error; see RFC 0821 )
	 EAT-ALL 
	S> .REPORT-ERRORS ;P 

-- -----------------------------------------------------------------------------------------------------------------------

  #25 =: IPPORT_SMTP	PRIVATE				-- standard SMTP port address
 #119 =: IPPORT_NNTP	PRIVATE				-- standard NNTP port address

CREATE  realname	PRIVATE ," Marcel Hendrix"
CREATE  mfrom	  	PRIVATE ," mhx@iae.nl"	 	-- from address
CREATE  newsserver	PRIVATE ," news.IAEhv.nl"
CREATE  server		PRIVATE ," smtp.IAEhv.nl"	-- server {name | dotted address}
CREATE  msubject	PRIVATE #256 CHARS ALLOT	-- msg subject
CREATE  mto	 	PRIVATE #256 CHARS ALLOT	-- to address

: (init) ( server port# "to" "subject" "file" -- )
	>S >S
	&" PARSE 2DROP  &" <WORD> mto      PACK DROP 
	&" PARSE 2DROP  &" <WORD> msubject PACK DROP
	&" PARSE 2DROP  &" <WORD> R/O BIN OPEN-FILE ?FILE TO file
	S> COUNT S> $OPEN-I/O [sock]  
	0 SET-SOCKET-TIMEOUT  EAT-ALL ;P

: (exit) ( -- ) file CLOSE-FILE DROP  SCLOSE-I/O [sock] ;P

: fsendline ( c-addr u -- ) +CR 2DUP .debug FSEND$ ;P
: sendline  ( c-addr u -- ) +CR 2DUP .debug SEND$ EAT-ALL ;P

: send-body ( -- )
	BEGIN  PAD DUP #1024 file READ-LINE ?FILE 0<> 
	WHILE  fsendline
   	REPEAT 2DROP 
	S" " +CR S" ." $+ sendline ;P

: helo-server ( -- ) S" HELO "     server COUNT  $+  cmd ;P
: mail-from   ( -- ) S" MAIL FROM:" mfrom COUNT <$+> cmd ;P
: rcpt-to     ( -- ) S" RCPT TO:"     mto COUNT <$+> cmd ;P
: rcpt-to-me  ( -- ) S" RCPT TO:"   mfrom COUNT <$+> cmd ;P
: data 	      ( -- ) S" DATA" cmd ;P
: send-quit   ( -- ) S" QUIT" cmd ;P
: mail-init   ( "to" "subject" "file" -- ) server IPPORT_SMTP (init) ;P
: mail-exit   ( -- ) send-quit (exit) ;P

: send-file ( -- )
	S" From: "    realname COUNT $+ S"  " $+ mfrom COUNT <$+> fsendline
	S" Subject: " msubject COUNT $+ fsendline
	S" To: "      mto      COUNT $+ fsendline
	send-body ;P
   
: mail-loop ( -- ) helo-server mail-from rcpt-to rcpt-to-me data send-file ;P
: SENDMAIL  ( "to" "subject" "file" -- )  mail-init  mail-loop  mail-exit ;

-- -------------------------------------------------------------------------------------------------------------------------

: mangled.t&d ( -- c-addr u ) 
	<# TIME&DATE  $DEADBEEF XOR U>D $DEADBEEF XOR # #  # #  
	5 0 DO  2DROP $DEADBEEF XOR U>D $DEADBEEF XOR # # LOOP #> ;P

-- This DayOfWeek program was posted by sakamoto@sm.sony.co.jp (Tomohiko Sakamoto) 
-- on comp.lang.c on March 10th, 1993. Sunday == 0.
: DoW(TS) ( d m y -- u )
	OVER 3 < + DUP 4 / 2DUP #100 / SWAP #100 / - + +
	SWAP 1+ CHARS C" -bed=pen+mad." + C@ + + 7 MOD ;P

-- Date and time in the required format.
: >newsdate
	DATE LOCALS| y m d | 
	d m y DoW(TS) >S C" Sun, Mon, Tue, Wed, Thu, Fri, Sat, " CHAR+ S> 5 * + 5 
	C" Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec " CHAR+ m 1- 4 * + 4 $+
	d (0DEC.R) $+ S"  " $+ 
	'TIME$     $+ S"  " $+ 
	y (0DEC.R) $+ S"  GMT" $+ ;P

: "Path:" 	  S" Path: not-for-mail" ;P
: "Newsgroups:"	  S" Newsgroups: " mto COUNT $+ ;P
: "Subject:"	  S" Subject: " msubject COUNT $+ ;P
: "From:"	  S" From: " mfrom COUNT $+ S"  (" $+ realname COUNT $+ S" )" $+ ;P
: "Message-ID:"   S" Message-ID: <" mangled.t&d $+ S" @frunobulax.edu>" $+ ;P
: "Date:"	  S" Date: " >newsdate $+ ;P
: "X-Newsreader:" S" X-Newsreader: iForth 2.0 console (18 December 2002)" ;P

: send-posting ( -- )
	"Path:"		  fsendline
	"Newsgroups:"     fsendline
	"Subject:"        fsendline
	"From:"	   	  fsendline
	"Message-ID:"     fsendline
	"Date:"	          fsendline
	"X-Newsreader:"   fsendline
	S" "		  fsendline
	send-body ;P

: post-init   ( "groups" "subject" "file" -- ) newsserver IPPORT_NNTP (init) ;P
: post-this   ( -- ) S" POST " cmd  send-posting ;P
: post-exit   ( -- ) send-quit (exit) ;P

: POST ( "groups" "subject" "file" -- )  post-init  post-this  post-exit ;

-- -------------------------------------------------------------------------------------------------------------------------

:ABOUT	CR .~ Usage: SENDMAIL "to_address" "subject" "messagefile"~ 
	CR .~    or: POST "to_groups" "subject" "postingfile"~
	CR
	CR .~ Example1: SENDMAIL "john@foo.bar" "Send mail works!" "file.txt"~ 
	CR .~ Example2: POST "comp.lang.forth" "Post news works!" "posting.txt"~ ;

                .ABOUT -pmail CR
		DEPRIVE
					(* End of Source *)