(*
 * LANGUAGE    : ANS Forth
 * PROJECT     : Forth Environments
 * DESCRIPTION : Get news using NNTP, see RFC-0977
 * CATEGORY    : Utility 
 * AUTHOR      : Marcel Hendrix 
 * LAST CHANGE : Wednesday, December 18, 2002 3:04 AM, Marcel Hendrix 
 * LAST CHANGE : Tuesday, December 24, 2002 8:40 PM, Marcel Hendrix; sstream
 * LAST CHANGE : Wednesday, December 25, 2002 1:42 AM, Marcel Hendrix; multiple group
 *)

	NEEDS -miscutil
	NEEDS -sstream

	REVISION -news " News with NNTP      Version 1.03 "

	PRIVATES

DOC
(*
   Use socket API functions to perform news acquisition using NNTP. 
   Connect to an (internally specified) NNTP server. Write new news
   to 'newsfile'. The article-last-read pointer is in 'markerfile'.
*)
ENDDOC

-- standard tools --------------------------------------------------------------------------------------------------------

CREATE eom$		PRIVATE	5 C, ^M C, ^J C, '.' C, ^M C, ^J C,
CREATE tst$		PRIVATE 5 C, 5 CHARS ALLOT
0 VALUE file		PRIVATE

SSTREAM [sock]		PRIVATE
CREATE  perr 		PRIVATE	   3 CHARS ALLOT	-- buffers error number from server
CREATE  response$	PRIVATE	#256 CHARS ALLOT	-- buffers response from server
0 VALUE #resp		PRIVATE				-- response size

VARIABLE debug? debug? OFF

: .debug  ( c-addr u -- ) DUP debug? @ AND IF  ^M EMIT '[' EMIT .TIME ." ] " TYPE EXIT  ENDIF 2DROP ;P
: <semit> ( char -- )     DUP EOF = IF  DROP ." <EOF>"  ELSE  EMIT  ENDIF ;P
: ?emit   ( char -- )     debug? @ IF  <semit> EXIT  ENDIF DROP ;P
: FSEND$  ( c-addr u -- ) 0 ?DO  C@+ SPUTCH [sock]  LOOP DROP ;P ( do a timely flush! )
: SEND$   ( c-addr u -- ) FSEND$  SFLUSH [sock] ;P
: CHAR1   ( -- c ) 	  BEGIN  SGETCH [sock] DUP EOF =  WHILE  DROP  REPEAT ;P
: !char	  ( c -- )	  response$ #resp +  C!  #resp 1+ #255 AND TO #resp ;P

: EAT-ALL ( -- )
	CLEAR #resp
	CHAR1 DUP ?emit !char 
	BEGIN  SGETCH [sock] DUP ?emit DUP !char DUP EOF = SWAP ^J = OR  UNTIL ;P

-- Errors are in perr. It is not necessary to translate them.
: .REPORT-ERRORS ( bool -- ) 
	0= ?EXIT
	CR ." Error " perr 3 TYPE SPACE response$ #resp TYPE
	TRUE ABORT" protocol error" ;P

: cmd? ( c-addr u -- TRUE=error )
	+CR 2DUP .debug SEND$
	CHAR1         DUP ?emit  perr           C!
	SGETCH [sock] DUP ?emit  perr   CHAR+   C!
	SGETCH [sock] DUP ?emit  perr 2 CHARS + C!
	perr C@ '3' > >S ( > 4xx is protocol error; see RFC 0977 )
	 EAT-ALL 
	S> ;P 

: cmd ( c-addr u -- ) cmd? .REPORT-ERRORS ;P 

: $>NUM ( c-addr1 u1 -- c-addr2 u2 u3 )
	BL SKIP  BL Split-At-Char  2SWAP 
	BASE @ >S  DECIMAL  NUMBER?  S> BASE !
	1 <> ABORT" $>num :: invalid number" ;P

: CLEAR-PATTERN ( -- ) tst$ CHAR+ tst$ C@ ERASE ;P

: END? ( char -- )
	tst$ 2+ tst$ 1+ tst$ C@ CMOVE  
	tst$ COUNT + 1- C!
	eom$ COUNT tst$ COUNT COMPARE 0= ;P

: separator ( c-addr u -- )
	$CR COUNT    S"  ---------------*----------------- " $+ 
	2SWAP $+     S"  ---------------*----------------- " $+ 
	+CR +CR file WRITE-FILE ?FILE ;P

: ?store-it ( bool c-addr u -- )
	0 LOCAL p
	separator
	?EXIT ( not available)
	CLEAR-PATTERN
	BEGIN
	  SGETCH [sock] 
	  DUP PAD p + C!  1 +TO p
	  p #4096 > IF  PAD p file WRITE-FILE ?FILE CLEAR p  ENDIF
	  END?
	UNTIL
	PAD p file WRITE-FILE ?FILE ;P

-- -----------------------------------------------------------------------------------------------------------------------

 #119 =: IPPORT_NNTP	PRIVATE	-- standard NNTP port address

CREATE server		PRIVATE ," news.IAEhv.nl"
CREATE markerfile	PRIVATE ," ./idata/newsrc.bin"

CREATE newsmarker	PRIVATE	0 ,
0 VALUE GROUPNAME	PRIVATE
0 VALUE NEWSFILE	PRIVATE
0 VALUE SLOT		PRIVATE
0 VALUE NEXTGROUP	PRIVATE
0 VALUE groupchain	PRIVATE

: NEWSGROUP ( slot# "name" "database" -- )
	CREATE	( slot) , HERE >S  0 , 0 , groupchain ,
		HERE S        !  BL WORD C@ 1+ ALLOT
		HERE S> CELL+ !  BL WORD C@ 1+ ALLOT
		@LATEST HEAD> @ TO groupchain
	DOES>	@+ TO SLOT
		@+ TO GROUPNAME
		@+ TO NEWSFILE 
		@  TO NEXTGROUP ;P

-- Define wanted newsgroups -----------------------------------------------------------------

	0 NEWSGROUP clf  comp.lang.forth 	./idata/clfnews.dat
	1 NEWSGROUP sci  sci.math.num-analysis  ./idata/scinews.dat

-- ------------------------------------------------------------------------------------------

: get^ ( -- u ) 
	markerfile COUNT R/W BIN OPEN-FILE ?FILE >S
	S FILE-SIZE ?FILE DROP SLOT CELLS CELL+
	U< IF 0
	 ELSE SLOT CELLS U>D S REPOSITION-FILE ?FILE
	      newsmarker =CELL S READ-FILE ?FILE DROP  newsmarker @
	ENDIF
	S> CLOSE-FILE DROP ;P

: put^ ( u -- ) 
	markerfile COUNT R/W BIN OPEN-FILE ?FILE >S
	SLOT CELLS U>D S REPOSITION-FILE ?FILE
	newsmarker !  
	newsmarker =CELL S WRITE-FILE ?FILE  S> CLOSE-FILE DROP ;P

: open-newsfile ( -- )
	NEWSFILE COUNT R/W BIN OPEN-FILE ?FILE TO file
	file FILE-SIZE ?FILE 	( move to eof )
	file REPOSITION-FILE ?FILE ;P

: close-newsfile ( -- )	file CLOSE-FILE DROP ;P

: nntp-init ( -- )
	server COUNT IPPORT_NNTP $OPEN-I/O [sock] 
	0 SET-SOCKET-TIMEOUT  EAT-ALL ;P

: nntp-exit ( -- ) S" QUIT" cmd  SCLOSE-I/O [sock] ;P

: fetch ( c-addr u -- first last #num ) 
	1 1 0 LOCALS| #articles first-article last-article |
	S" GROUP " 2SWAP $+ cmd  
	response$ #resp
	$>NUM TO #articles  
	$>NUM TO first-article  
	$>NUM TO last-article 2DROP
	CR #articles DEC. ." articles, from " first-article DEC. ." to " last-article 0 .R
	first-article get^ MAX  DUP ." , read pointer = " DEC.
	last-article   2DUP SWAP - 1+ 
	CR ." Retrieving " DUP DEC. ." article" DUP ?s ;P

: ask-for-it? ( u -- TRUE=error ) S" ARTICLE " ROT (0DEC.R) $+ cmd? ;P

: (store) ( first last -- )
	  DUP >S ( save last )
	   1+ SWAP ?DO  I ask-for-it? S" NEXT ARTICLE" ?store-it  LOOP
	  S> 1+ put^ ;P

: store-group ( -- )    
	GROUPNAME COUNT fetch ( -- first last #num )
	0= IF  2DROP CR ." No new news in " GROUPNAME .$ EXIT  ENDIF
	open-newsfile
	 ( first last -- ) 
	 ['] (store) CATCH ?DUP IF close-newsfile THROW ENDIF
	close-newsfile 
	CR GROUPNAME .$ ." 's new articles stored." ;P

: store-news ( -- ) 
	groupchain EXECUTE 
	BEGIN  
	  store-group 
	  NEXTGROUP ?DUP 
	WHILE  
	  EXECUTE 
	REPEAT ;P

: NEWS	( -- ) nntp-init store-news nntp-exit ;

:ABOUT	CR ." Usage: NEWS -- get news to newsfiles" ;

                .ABOUT -news CR
		DEPRIVE
		NEWS

				(* End of Source *)